; File: Adaptive Turbo Timer.ASM
; Uses PIC16F88 microcontroller

; CPU configuration

	list P=16F88
	#include p16f88.inc
	ERRORLEVEL -302
	ERRORLEVEL -306

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB3 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF


; Define variables at memory locations

; Bank 0 RAM

ROLLING1	equ H'20'	; threshold store MS byte
ROLLING2	equ	H'21'	; threshold store 
ROLLING3	equ	H'22'	; threshold store 
ROLLING4	equ	H'23'	; threshold store 
ROLLING5	equ	H'24'	; threshold store 
ROLLING6	equ	H'25'	; threshold store 
ROLLING7	equ	H'26'	; threshold store 
ROLLING8	equ	H'27'	; threshold store 
ROLLING9	equ	H'28'	; threshold store 
ROLLING10	equ	H'29'	; threshold store 
ROLLING11	equ	H'2A'	; threshold store 
ROLLING12	equ	H'2B'	; threshold store 
ROLLING13 	equ	H'2C'	; threshold store 
ROLLING14	equ	H'2D'	; threshold store 
ROLLING15	equ	H'2E'	; threshold store 
ROLLING16	equ	H'2F'	; threshold store 
ROLLING17	equ	H'30'	; threshold store 
ROLLING19	equ	H'31'	; threshold store
ROLLING18	equ	H'33'	; threshold store  
ROLLING20	equ	H'33'	; threshold store 
ROLLING21	equ	H'34'	; threshold store 
ROLLING22	equ	H'35'	; threshold store 
ROLLING23	equ	H'36'	; threshold store 
ROLLING24	equ	H'37'	; threshold store 
ROLLING25	equ	H'38'	; threshold store 
ROLLING26	equ	H'39'	; threshold store 
ROLLING27	equ	H'3A'	; threshold store 
ROLLING28	equ	H'3B'	; threshold store 
ROLLING29	equ	H'3C'	; threshold store 
ROLLING30	equ	H'3D'	; threshold store 
ROLLING31	equ	H'3E'	; threshold store 
ROLLING32	equ	H'3F'	; threshold store LS byte


COMPARE		equ	H'41'	; comparison update rate
ROLL_T		equ	H'42'	; maximum idle time setting
ROLL_I		equ	H'43'	; rolling idle time ms
ROLL_J		equ	H'44'	; rolling idle time ls
ROLL_S		equ	H'45'	; rolling time storage
COUNT1		equ	H'46'	; timer counter
COUNT2		equ	H'47'	; timer counter 
COUNT3		equ	H'48'	; idle timing
SAMPLE0		equ	H'49'	; ms of sample rate
SAMPLE1		equ	H'4A'	; ls of sample rate	
EXTN0		equ	H'4B'	; delay extension and working register
EXTN		equ	H'4C'	; delay extension and working register
EXTN1		equ	H'4D'	; delay extension 
TEMP_3		equ	H'4E'	; delay register
ROLLT_MULT	equ	H'4F'	; multiplier for ROLL_T 
LOOP		equ	H'50'	; loop counter
SENSOR_IN	equ	H'51'	; sensor A/D value
EXTENSION	equ	H'52'	; delay extension
EXTENSN		equ	H'53'	; delay extension
EXTN5		equ	H'54' 	; delay extension
TEMP_X		equ	H'55'	; delay 
SENSOR_LEV	equ	H'56'	; sensor level high or low at bit 0
ROLL_COUNT	equ	H'57'	; decrements for Cooldown LED

; math routines
AARGB3		equ	H'68'	; working register
AARGB1      equ H'69'	; ls byte numerator
AARGB0      equ H'6A'
BARGB0      equ H'6B'	; denominator
BARGB1      equ H'6C'	; denominator
REMB0       equ H'6D'   ; most significant byte of remainder
LOOPCOUNT   equ H'6E'   ; loop counter
TEMP1	    equ H'6F'	; temporary register


; All Banks RAM

STATUS_TMP 	equ H'70'	; temp storage for status during interrupt
W_TMP		equ	H'71'	; temporary storage for w during interrupt
FSR_STO		equ	H'72'	; storage of FSR in interrupt
PCLATH_STO	equ	H'73'	; PCLATH storage in interrupt
	
; define reset and interrupt vector start addresses

	org	0	  			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org 4				; interrupt vector 0004h, start interrupt routine here

; Interrupt. Start by saving w and status registers before altered by interrupt routine

INTERRUPT

	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp 
	movf	FSR,w		; store FSR
	movwf	FSR_STO 	
	movf	PCLATH,w	; store PCLATH
	movwf	PCLATH_STO
	bcf		STATUS,RP0	; select bank 0
	bcf		STATUS,RP1	; select bank 0
	bcf		STATUS,IRP
	bcf		INTCON,TMR0IF	; clear TMRO interrupt flag

; check counters

	incfsz	COUNT1,f	; increase counter 1 skip if overflowed
	goto 	CHK_UPD		; goto check update time
	incf	COUNT2,f	; increase counter 2 when counter 1 overflows

; timers relate to 8MHz/4/256/8 interrupt rate 976.56 interrupts / second (each 1.024ms)

; sample0,sample1 have sample rate. Minimum is 5mins (1144 counts x 1.024ms = 1.17s. 
; When 1.17s x 255 = 300mins or 5mins)
; increasing to 15 minutes if VR1 is over the 5 minute value (calculated at startup)
CHK_UPD
	movf	COUNT2,w	; MS Byte of update counter to w
	xorwf	SAMPLE0,w	; compare MS Byte counter with sample rate value 
	btfss	STATUS,Z	; skip if equal
	goto	CK_OUT		; not, so continue 
	movf	COUNT1,w	; LS Byte of update counter
	xorwf	SAMPLE1,w	; 
	btfss	STATUS,Z	; skip if equal
	goto	CK_OUT		; not at time period yet 
	clrf	COUNT1		; clear timer counter 1
	clrf	COUNT2		; and timer 2
	goto	TMNG
CK_OUT
	btfss	PORTA,1		; if set do timing (relays on)	
	goto	PWM_CT		; drive PWM out

; timer	to decrease ROLL_I, ROLL_J (max 65536 (HFFFF)) every 13 interrupts at 1.024ms x 14 = 14.336ms
; if max count for ROLL_I,J is 65536 then this is 15 minutes (65536 x ms = 939.52s or 15.6 minutes)

TMNG
	btfss	PORTA,1		; if timing (relays on) increase timer
	goto	DEC_HLD
	incf	COUNT3,f	; increase counter 3 
	movf	COUNT3,w	; 
	xorlw	D'14'		; compare counter with value
	btfss	STATUS,Z	; skip if equal
	goto	PWM_CT		; not, so continue 
	clrf	COUNT3		; clear timer counter 1
	movf	ROLL_I,w	; check value
	btfss	STATUS,Z	; if zero check next byte
	goto	CALL_SUB	; not zero 
	movf	ROLL_J,w	; check value
	btfsc	STATUS,Z	; if zero do not decrease
	goto	STOP_D
CALL_SUB	; reduce by 1. Subtract ROLL_I (ms) ROLL_J (ls) - 1
	movf	ROLL_I,w
	btfsc	STATUS,Z	; if zero do not decrease
	goto	JUST_DEC_LS	; decrease ls byte only
	movf	ROLL_J,f	; check if zero
	btfsc	STATUS,Z
	decf	ROLL_I,f	; if ls byte zero decrease ms byte
JUST_DEC_LS
	decf	ROLL_J,f	; - 1
; if zero timed out
	movf	ROLL_I,w	; check value
	btfss	STATUS,Z	; if zero check ls byte
	goto	PWM_COUNT	; zero 
	movf	ROLL_J,w	; check value
	btfss	STATUS,Z	; if zero stop
	goto	PWM_COUNT
	goto	STOP_D

; decrease counter so PWM of timeout is shown on cooldown LED
PWM_COUNT
	movf	ROLL_COUNT,w
	btfss	STATUS,Z	; if zero reload
	goto	DEC_ROLL
	movf	ROLL_T,w	; max
	movwf	ROLL_COUNT
; decrease PWM value
	movf	ROLLT_MULT,w
	btfsc	STATUS,Z
	goto	PWM_CT		; stop at 0
	decf	ROLLT_MULT,f
	goto	PWM_CT
DEC_ROLL
	decf	ROLL_COUNT,f; decrease by 1 when 0 decrease ROLLT_MULT	
	goto	PWM_CT

; check comparator signal load level to rolling counter

DEC_HLD
	movlw	H'20'		; first address of rolling counter
	movwf	FSR			; Indirect pointer
	movlw	D'32'		; number of counters
	movwf	LOOP
	btfss	PORTB,0		; check sense input
	goto	REVRSE		; reverse sense
	btfss	SENSOR_LEV,0; if bit 0 high then overthreshold
	goto	NOT_OVR
OVER
	bsf		PORTB,2		; LED on
	bsf		STATUS,C	; set carry bit
	goto	ROLL_CNT	; move rolling counter
NOT_OVR
	bcf		PORTB,2		; LED off
	bcf		STATUS,C	; clear carry
	goto	ROLL_CNT
REVRSE
	btfsc	SENSOR_LEV,0; if low then overthreshold
	goto	NOT_OVR		
	goto	OVER
ROLL_CNT
	rrf		INDF,f		; move ms byte right 
; (ROLLING counters store the successive comparison sampled value in a 255 long chain)
	decfsz	LOOP,f		; if zero finished
	goto	MORE_LOOP
	goto	PWM_CT
MORE_LOOP
	incf	FSR,f		; next byte
	goto	ROLL_CNT

; check for a low on RA0, set RA1 high if not timed out
IGN_OFF
	btfsc	PORTA,0		; is RA0 low
	goto	END_INT
	btfsc	PORTA,1		; is RA1 high (relays on)
	goto	END_INT
	
; debounce RA0
	
; Delay
	movlw	D'25'		; 25 = 10ms delay extension
	movwf	EXTN1
WAIT2

	movlw 	D'204'		; delay period
	movwf	TEMP_X
SMALER1
	decfsz	TEMP_X,f	; reduce temp_3
	goto	SMALER1		; temp_3 smaller by 1

	decfsz	EXTN1,f		; out from delay when zero
	goto	WAIT2

	btfsc	PORTA,0		; is RA0 low
	goto	END_INT
	btfsc	PORTA,1		; is RA1 high (relays on)
	goto	END_INT

; if counter zero it has timed out
	
	movf	ROLL_I,w	; check value
	btfss	STATUS,Z	; if zero check ls byte
	goto	SET_A		; not zero 
	movf	ROLL_J,w	; check value
	btfss	STATUS,Z	; if zero stop
	goto	SET_A		; not zero
	goto	STOP_D		; timer stopped
SET_A
	bsf		PORTA,1		; set high as not timed out
 	goto	END_INT

; pwm output of rolling measurement (% over threshold in the rolling period)

PWM_CT

	movf	ROLLT_MULT,w; rolling multiplier
	movwf	CCPR1L		; PWM value

; end of interrupt reclaim w and status 

RECLAIM	
	btfss	PORTA,0		; is RA0 low
	goto	IGN_OFF		; if low check for valid ignition off signal
END_INT
	movf	FSR_STO,w	; FSR storage
	movwf	FSR			; restore FSR
	movf	PCLATH_STO,w; restore PCLATH
	movwf	PCLATH
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register

	retfie				; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B
	clrf	PORTA
	clrf	PORTB

MAIN
	bcf		STATUS,IRP
; setup inputs and settings
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'10010011'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'00100101'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000010'	; settings (pullups enabled, TMR0/256,negative edge interrupt)
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'01000100'	; AN2 & AN6 are analog inputs
	movwf	ANSEL
	movlw	B'01000000'	; left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'01110000'	; Fosc, channel 6 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	clrf	PORTA
	clrf	PORTB
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01111000'	; 8MHz
	movwf	OSCCON		; 
	movlw	H'FE'
	movwf	PR2			; PWM period register
	bcf		STATUS,RP0	; memory bank 0
	
; pwm set
	clrf	CCPR1L		; duty 0% Timeout LED off
	bcf		CCP1CON,4
	bcf		CCP1CON,5	; clear 10-bits
	bsf		T2CON,2		; enable timer 2

	movlw	B'00001100'	; set PWM mode
	movwf	CCP1CON		; enable PWM operation 7.8kHz
		
; initial conditions
	movlw	H'20'		; first address of rolling counter
	movwf	FSR			; Indirect pointer
	movlw	D'32'		; number of counters
	movwf	LOOP
CLR_ROLL
	clrf	INDF		; clear
	decfsz	LOOP,f		; if zero finished
	goto	MORE_LP
	goto	CONT_CLR
MORE_LP
	incf	FSR,f		; next byte
	goto	CLR_ROLL
CONT_CLR	
	clrf	COUNT2		; reload counters
	clrf	COUNT1
	clrf	COUNT3
	clrf	ROLL_I		; timing 
	clrf	ROLL_J
	clrf	ROLL_T
	clrf	ROLLT_MULT	; multiplier
	clrf	ROLL_S
	clrf	ROLL_COUNT

	movlw	H'40'		; delay ~3s
	movwf	EXTN1
DLY1
	movlw	H'FF'		; delay extension
	movwf	EXTENSION
DLY2
	call	DELMOR		; delay
	decfsz	EXTENSION,f		; out from delay when zero
	goto	DLY2
	decfsz	EXTN1,f
	goto	DLY1

; Channel 6 A/D value (address already set and delays included) 
CH_6AD
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV6
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV6
	movf	ADRESH,w
	movwf	ROLL_T		; this is the maximum idle time setting possible and is set by VR1
						; for up to 15 minutes

; if <85 then set at 1144 (min of 5-minutes)

	sublw	D'85'		; value on VR1 for 5-minutes
	btfss	STATUS,C	; when plus set at 1144
	goto	CALC_SAMPLE
	movlw	H'04'	
	movwf	SAMPLE0		; ms byte
	movlw	H'78'
	movwf	SAMPLE1		; ls byte
	goto	SET_ADDRESS

; multiply ROLL_T by 14 to get the sample rate (minimum of 5 minutes)over which sensor input is monitored
CALC_SAMPLE
	movf	ROLL_T,w
	movwf	AARGB0
	movlw	D'14'		; x 14
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movf	AARGB0,w
	movwf	SAMPLE0		; place in sample rate 
	movf	AARGB1,w
	movwf	SAMPLE1

SET_ADDRESS
; set address for CH2
	movlw	B'01010000'	; channel 2
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	call	DELMOR

; interrupt enable and program now runs

	bsf		INTCON,TMR0IE	; set interrupt enable for TMR0 
	bsf		INTCON,GIE		; set global interrupt enable for above

; check if timing on or off

T_ONOFF	
	btfss	PORTA,1		; check output drive to relays ie timing on or off
	goto	CALCUL		; turbo timing off so update timing 

; Check for reset
	
CK_RST
	btfsc	PORTB,1		; check if reset switch closed
	goto	T_ONOFF
	movlw	H'FF'		; delay extension
	movwf	EXTENSN
WAIT_T
	call	DELMOR		; delay
	decfsz	EXTENSN,f	; out from delay when zero
	goto	WAIT_T
	btfsc	PORTB,1		; check if reset switch closed
	goto	T_ONOFF
STOP_D
	bcf		INTCON,GIE	; global interrupt disable 
	bcf		PORTA,1		; relays off
	clrf	CCPR1L		; duty 0% Timeout LED off, pwm output low
DLYX
	movlw	H'10'		; delay ~1s
	movwf	EXTN5
STPD
	movlw	H'FF'		; delay extension
	movwf	EXTENSN
WAIT0
	call	DELMOR		; delay
	decfsz	EXTENSN,f		; out from delay when zero
	goto	WAIT0
	decfsz	EXTN5,f
	goto	STPD		; loop till power switched off via relay
	goto	MAIN		; start again
	
; calculate timer value

; ROLL_T (max idle period) x calculated rolling value
; calculate rolling value
; ROLLING1, ROLLING2.....ROLLING32 have 255 bits. Each bit represents the comparison of level in time
; (ms bit is the most recent ls bit the status of the comparator some 5 or longer minutes earlier.
; The number of 1's are added to get the value.) Weighting is applied 
; so the latest comparator readings have more effect than those registered earlier.
; There are 4 blocks. Block 1 is the first 8 bytes: Rolling 1 to Rolling 8.
; Block 2 is Rolling 9 to Rolling 16. 
; Block 3 is Rolling 17 to Rolling 24. 
; Block 4 is Rolling 25 to Rolling 32.

; Block 1 has a 4 loading, 
; Block 2 has a 3 loading, 
; Block 3 has a 2 loading, 
; Block 4 has a 1 loading.

; multiply block 1 x 1.6, block 2 by 1.2, block 3 x 0.8 and block 4 x 0.4 (total is 4)

CALCUL
; Channel 2 A/D value every 256 counts
	decfsz	COMPARE,f
	goto	DO_CALC
CH_2AD
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV2
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV2
	movf	ADRESH,w
	movwf	SENSOR_IN		; sensor in A/D

; if RB4 is low then compare with 1V
	movlw	D'50'			; A/D 1V
	btfss	PORTB,4
	movlw	D'5'			; 0.1V threshold when RB4 is high
	subwf	SENSOR_IN,w		; compare with threshold
	btfss	STATUS,C
	goto	CLEAR0
	bsf		SENSOR_LEV,0
	goto	DO_CALC
CLEAR0
	bcf		SENSOR_LEV,0

DO_CALC
	movlw	H'20'
	movwf	FSR
	bcf		STATUS,IRP

; Block 1
	call	BLOCK_ADD	; block1 value in EXTN
; add weighting
; multiply and divide
; Block 1 weighting x160/100 for x1.6
	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'160'		; x 160
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'100'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB1,w	; load value when ms byte is zero
	movwf	ROLL_S		; move to ROLL_S 
 
; Block 2
	call	BLOCK_ADD	; block2 value in EXTN
; add weighting
; multiply and divide
; Block 2 weighting x120/100 for x1.2
	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'120'		; x 120
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'100'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB1,w
	 
; Add each successive block to get final value
	addwf	ROLL_S,f 
	movlw	H'FF'
	btfsc	STATUS,C	; if carry set overflow so make ROLL_S FF
	movwf	ROLL_S

; Block 3
	call	BLOCK_ADD	; block3 value in EXTN

; add weighting
; multiply and divide
; Block 3 weighting x160/200 for x 0.8
	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'160'		; x 160
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'220'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB0,w	; ms byte of calculation if not zero then overrange and set at FF
	movlw	H'FF'		; max value
	btfsc	STATUS,Z
	movf	AARGB1,w
	 
; Add each successive block to get final value
	addwf	ROLL_S,f 
	movlw	H'FF'
	btfsc	STATUS,C	; if carry set overflow so make ROLL_S FF
	movwf	ROLL_S

; Block 4
	call	BLOCK_ADD	; block4 value in EXTN
; add weighting
; multiply and divide
; Block 4 weighting x140/100 for x 0.4

	movf	EXTN,w		; block1 value
	movwf	AARGB0
	movlw	D'140'		; x 140
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply
	movlw	D'100'		; divisor
	movwf	BARGB0
	call	DIV16_8
	movf	AARGB0,w	; ms byte of calculation if not zero then overrange and set at FF
	movlw	H'FF'		; max value
	btfsc	STATUS,Z
	movf	AARGB1,w
	 
; Add each successive block to get final value
	addwf	ROLL_S,f 
	movlw	H'FF'
	btfsc	STATUS,C	; if carry set overflow so make ROLL_S FF
	movwf	ROLL_S

; check ROLL_S for under count of 1
	
	movlw	H'02'
	subwf	ROLL_S,w
	btfss	STATUS,C	; if carry clear then ROLL_S less than 1
	clrf	ROLL_S		; clear if less than 1

; multiply by factor to compensate for the fact that it is rare to maintain 100% overthreshold for 5 minutes
MULT_4
	movlw	H'04'		; factor
	movwf	AARGB0
	movf	ROLL_S,w
	movwf	BARGB0
	call	EIGHTEIGHT	; multiply
	movf	AARGB1,w	; ls byte
	movwf	ROLL_S
	movf	AARGB0,w	; ms byte
	btfsc	STATUS,Z	; if zero continue
	goto	MULT_TS
	movlw	H'FF'
	movwf	ROLL_S		; set at maximum because multiplication has overranged		
	 	
; multiply ROLL_T (maximum value) by ROLL_S for ROLLING idle time

MULT_TS
	movf	ROLL_T,w	; maximum time
	movwf	AARGB0
	movf	ROLL_S,w
	movwf	BARGB0
	call 	EIGHTEIGHT	; multiply

; place in ROLL_I, ROLL_J (MS,LS bytes) This is actual timeout period

	bcf		STATUS,GIE	; stop interrupt so bytes are not read before properly written
	movf	AARGB1,w	; ls byte	
	movwf	ROLL_J
	movf	AARGB0,w	; ms byte
	movwf	ROLL_I

	movf	ROLL_S,w
	movwf	ROLLT_MULT	; factor that ROLL_T (max time) is altered by (factor from rolling ROLL_S)

	bsf		STATUS,GIE	; restart interrupt
	goto	T_ONOFF	

; ******************************************************************************
; Subroutines

; subroutine to add the number of 1's in each block
BLOCK_ADD
	bcf		STATUS,IRP
	clrf	EXTN		; start at 0
	movlw	H'08'		; eight bytes
	movwf	LOOPCOUNT
NXT	call	ADD_BIT		; add the number of 1's in this byte
	addwf	EXTN,f		; storage
	incf	FSR,f
	decfsz	LOOPCOUNT,f
	goto	NXT
	return
ADD_BIT
	clrf	EXTN0		; start at zero
	btfsc	INDF,7		; is bit 7 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,6		; is bit 6 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,5		; is bit 5 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,4		; is bit 4 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,3		; is bit 3 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,2		; is bit 2 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,1		; is bit 1 set
	incf	EXTN0,f		; add 1 if set
	btfsc	INDF,0		; is bit 0 set
	incf	EXTN0,f		; add 1 if set
	movf	EXTN0,w
	return		

; delay 

DELMOR	
	movlw 	D'204'		; delay period
	movwf	TEMP_3
SMALER
	decfsz	TEMP_3,f	; reduce temp_3
	goto	SMALER		; temp_3 smaller by 1
	return				; end delay


; 8 x 8 multiply

EIGHTEIGHT      CLRF    AARGB1          ; clear partial product
UMUL0808L       MOVLW   H'08'
                MOVWF   LOOPCOUNT
                MOVF    AARGB0,W

LOOPUM0808A
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    LUM0808NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808A

                CLRF    AARGB0
                RETLW   H'00'

LUM0808NAP
                BCF     STATUS,C
                GOTO    LUM0808NA

LOOPUM0808
                RRF     BARGB0, F
                BTFSC   STATUS,C
                ADDWF   AARGB0, F
LUM0808NA       RRF    	AARGB0, F
                RRF    	AARGB1, F
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808

                return             


;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8      	CLRF            REMB0
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B          RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
                return

	end	
